<!-- Explosive Engine: Search Engine for FlatlyPage CMS -->
<?php
error_reporting(E_ALL);
ini_set('display_errors', 0);

header("X-Powered-By: FlatlyPage CMS");
header("X-Content-Type-Options: nosniff");
header("X-Frame-Options: SAMEORIGIN");
header("X-XSS-Protection: 1; mode=block");
header("Referrer-Policy: strict-origin-when-cross-origin");

$base_dir = dirname(__FILE__);
$data_dir = $base_dir . '/data';
$cache_dir = $base_dir . '/data/search';

define('BASE_DIR', $base_dir);
define('DATA_DIR', $data_dir);
define('CACHE_DIR', $cache_dir);
define('CACHE_FILE', $cache_dir . '/results.xml');
define('CACHE_TIME', 300);
define('RESULTS_PER_PAGE', 20);

$settings_file = DATA_DIR . '/settings.php';
$settings = file_exists($settings_file) ? include($settings_file) : [];
$site_name = isset($settings['site_name']) ? $settings['site_name'] : 'Find articles and pages';

function sanitize_query($query) {
    $query = trim($query);
    $query = strip_tags($query);
    $query = htmlspecialchars($query, ENT_QUOTES, 'UTF-8');
    return mb_substr($query, 0, 100);
}

function build_search_index() {
    $index = [];
    $pattern = DATA_DIR . '/product-*.php';
    $files = glob($pattern);
    
    if (!$files) {
        return $index;
    }
    
    foreach ($files as $file) {
        if (!is_file($file)) {
            continue;
        }
        
        $basename = basename($file);
        if (!preg_match('/^product-[a-z0-9\-]+\.php$/i', $basename)) {
            continue;
        }
        
        try {
            $content = file_get_contents($file);
            
            if (strlen($content) > 1048576) {
                continue;
            }
            
            if (preg_match("/'slug'\s*=>\s*'([^']+)'/", $content, $slug_match) &&
                preg_match("/'title'\s*=>\s*'([^']+)'/", $content, $title_match) &&
                preg_match("/'description'\s*=>\s*'([^']+)'/", $content, $desc_match)) {
                
                $slug = preg_replace('/[^a-z0-9\-]/', '', strtolower($slug_match[1]));
                $title = strip_tags($title_match[1]);
                $description = strip_tags($desc_match[1]);
                
                if (strlen($slug) > 0 && strlen($title) > 0) {
                    $index[] = [
                        'slug' => mb_substr($slug, 0, 200),
                        'title' => mb_substr($title, 0, 200),
                        'description' => mb_substr($description, 0, 500),
                        'file' => $basename
                    ];
                }
            }
        } catch (Exception $e) {
            continue;
        }
    }
    
    return $index;
}

function save_cache($index) {
    if (!is_dir(CACHE_DIR)) {
        if (!mkdir(CACHE_DIR, 0755, true)) {
            return false;
        }
    }
    
    if (!is_writable(CACHE_DIR)) {
        return false;
    }
    
    try {
        $xml = new SimpleXMLElement('<?xml version="1.0" encoding="UTF-8"?><search_index/>');
        $xml->addAttribute('generated', date('c'));
        $xml->addAttribute('version', '1.0');
        
        foreach ($index as $item) {
            $product = $xml->addChild('product');
            $product->addChild('slug', htmlspecialchars($item['slug'], ENT_XML1 | ENT_QUOTES, 'UTF-8'));
            $product->addChild('title', htmlspecialchars($item['title'], ENT_XML1 | ENT_QUOTES, 'UTF-8'));
            $product->addChild('description', htmlspecialchars($item['description'], ENT_XML1 | ENT_QUOTES, 'UTF-8'));
            $product->addChild('file', htmlspecialchars($item['file'], ENT_XML1 | ENT_QUOTES, 'UTF-8'));
        }
        
        $temp_file = CACHE_FILE . '.tmp.' . uniqid();
        $xml_content = $xml->asXML();
        
        if ($xml_content === false) {
            return false;
        }
        
        if (file_put_contents($temp_file, $xml_content, LOCK_EX) === false) {
            return false;
        }
        
        if (!rename($temp_file, CACHE_FILE)) {
            @unlink($temp_file);
            return false;
        }

        $txt_file = CACHE_DIR . '/search.txt';
        file_put_contents($txt_file, 'true');
        @chmod($txt_file, 0644);

        @chmod(CACHE_FILE, 0644);
        return true;
        
    } catch (Exception $e) {
        if (isset($temp_file) && file_exists($temp_file)) {
            @unlink($temp_file);
        }
        return false;
    }
}

function load_cache() {
    if (!file_exists(CACHE_FILE)) {
        return null;
    }
    
    if (time() - filemtime(CACHE_FILE) > CACHE_TIME) {
        return null;
    }
    
    if (filesize(CACHE_FILE) > 5242880) {
        return null;
    }
    
    try {
        libxml_use_internal_errors(true);
        
        $xml = simplexml_load_file(CACHE_FILE, 'SimpleXMLElement', LIBXML_NONET | LIBXML_NOCDATA);
        
        if (!$xml) {
            libxml_clear_errors();
            return null;
        }
        
        if (!isset($xml['version']) || (string)$xml['version'] !== '1.0') {
            return null;
        }
        
        $index = [];
        
        foreach ($xml->product as $product) {
            if (!isset($product->slug) || !isset($product->title)) {
                continue;
            }
            
            $slug = (string)$product->slug;
            $title = (string)$product->title;
            $description = isset($product->description) ? (string)$product->description : '';
            $file = isset($product->file) ? (string)$product->file : '';
            
            if (!preg_match('/^[a-z0-9\-]+$/i', $slug)) {
                continue;
            }
            
            $index[] = [
                'slug' => $slug,
                'title' => $title,
                'description' => $description,
                'file' => $file
            ];
        }
        
        return $index;
        
    } catch (Exception $e) {
        return null;
    }
}

function search_products($query, $index) {
    if (empty($query) || strlen($query) < 2) {
        return [];
    }
    
    $results = [];
    $query_lower = mb_strtolower($query, 'UTF-8');
    
    $stop_words = ['is', 'the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for'];
    $terms = array_filter(explode(' ', $query_lower), function($term) use ($stop_words) {
        return !empty($term) && strlen($term) > 1 && !in_array($term, $stop_words);
    });
    
    if (empty($terms)) {
        return [];
    }
    
    $terms = array_values($terms);
    $total_terms = count($terms);
    
    foreach ($index as $item) {
        $score = 0;
        $matched_terms = 0;
        $title_lower = mb_strtolower($item['title'], 'UTF-8');
        $desc_lower = mb_strtolower($item['description'], 'UTF-8');
        $slug_lower = mb_strtolower($item['slug'], 'UTF-8');
        $combined = $title_lower . ' ' . $desc_lower . ' ' . $slug_lower;
        
        if (stripos($combined, $query_lower) !== false) {
            $score += 100;
            if (stripos($title_lower, $query_lower) !== false) {
                $score += 50;
            }
        }
        
        for ($i = 0; $i < count($terms) - 1; $i++) {
            $bigram = $terms[$i] . ' ' . $terms[$i + 1];
            if (stripos($title_lower, $bigram) !== false) {
                $score += 25;
            }
            if (stripos($desc_lower, $bigram) !== false) {
                $score += 15;
            }
        }
        
        foreach ($terms as $term) {
            if (empty($term)) continue;
            
            $term_found = false;
            
            if (stripos($title_lower, $term) !== false) {
                $score += 10;
                $term_found = true;
            }
            
            if (stripos($desc_lower, $term) !== false) {
                $score += 5;
                $term_found = true;
            }
            
            if (stripos($slug_lower, $term) !== false) {
                $score += 2;
                $term_found = true;
            }
            
            if ($term_found) {
                $matched_terms++;
            }
        }
        
        $required_coverage = $total_terms <= 2 ? $total_terms : ceil($total_terms * 0.4);
        
        if ($matched_terms >= $required_coverage && $score > 0) {
            $coverage_ratio = $matched_terms / $total_terms;
            $score = $score * (0.5 + ($coverage_ratio * 0.5));
            
            $item['score'] = round($score);
            $item['matched_terms'] = $matched_terms;
            $item['coverage'] = $coverage_ratio;
            $results[] = $item;
        }
    }
    
    usort($results, function($a, $b) {
        if ($b['score'] == $a['score']) {
            return $b['coverage'] <=> $a['coverage'];
        }
        return $b['score'] - $a['score'];
    });
    
    return $results;
}

$query = isset($_GET['q']) ? sanitize_query($_GET['q']) : '';
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$results = [];
$total_results = 0;
$search_performed = !empty($query);

if ($search_performed) {
    $index = load_cache();
    
    if ($index === null) {
        $index = build_search_index();
        save_cache($index);
    }
    
    $all_results = search_products($query, $index);
    $total_results = count($all_results);
    
    $offset = ($page - 1) * RESULTS_PER_PAGE;
    $results = array_slice($all_results, $offset, RESULTS_PER_PAGE);
}

$has_more = $total_results > ($page * RESULTS_PER_PAGE);
$showing_from = $total_results > 0 ? (($page - 1) * RESULTS_PER_PAGE + 1) : 0;
$showing_to = min($total_results, $page * RESULTS_PER_PAGE);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php $page_title = $search_performed ? 'Search results for: ' . $query : 'Search our site'; echo htmlspecialchars($page_title . ' | ' . $site_name, ENT_QUOTES, 'UTF-8'); ?></title>
    <link rel="stylesheet" href="/css/styles.css">
    
    <?php 
    $themePath = __DIR__ . '/css/theme.css';
    $themeUrl = '/css/theme.css';

    if (is_file($themePath) && filesize($themePath) > 0) {
        
        $handle = fopen($themePath, 'r');
        $header = fread($handle, 25);
        fclose($handle);
        if (trim($header) !== '/* No theme active */') {
            $v = filemtime($themePath);
            echo '<link rel="stylesheet" href="' . htmlspecialchars($themeUrl) . '?v=' . $v . '">';
        }
    }
    ?>
    
    <style>
        .search-page {
            padding: 160px 0 100px;
            min-height: 70vh;
        }
        
        .search-header {
            text-align: center;
            margin-bottom: 48px;
        }
        
        .search-header h1 {
            font-size: 2.5rem;
            font-weight: 700;
            letter-spacing: -0.02em;
            margin-bottom: 16px;
        }
        
        .search-header p {
            color: var(--muted);
            font-size: 1.0625rem;
        }
        
        .search-form {
            max-width: 600px;
            margin: 0 auto 48px;
        }
        
        .search-input-wrapper {
            position: relative;
            display: flex;
            gap: 12px;
        }
        
        .search-input {
            flex: 1;
            padding: 16px 20px 16px 52px;
            border: 1px solid var(--border);
            border-radius: 12px;
            background: var(--card);
            color: var(--foreground);
            font-size: 1rem;
            font-family: inherit;
            transition: all 0.2s;
        }
        
        .search-input:focus {
            outline: none;
            border-color: var(--foreground);
            box-shadow: 0 0 0 3px rgba(255, 255, 255, 0.05);
        }
        
        .search-icon {
            position: absolute;
            left: 18px;
            top: 50%;
            transform: translateY(-50%);
            width: 20px;
            height: 20px;
            color: var(--muted);
            pointer-events: none;
        }
        
        .search-button {
            padding: 16px 32px;
            background: var(--foreground);
            color: var(--background);
            border: none;
            border-radius: 12px;
            font-weight: 600;
            font-size: 1rem;
            cursor: pointer;
            transition: all 0.2s;
            white-space: nowrap;
        }
        
        .search-button:hover {
            opacity: 0.9;
            transform: translateY(-2px);
        }
        
        .search-meta {
            text-align: center;
            color: var(--muted);
            font-size: 0.9375rem;
            margin-bottom: 32px;
        }
        
        .search-results {
            max-width: 900px;
            margin: 0 auto;
        }
        
        .result-card {
            background: var(--card);
            border: 1px solid var(--border);
            border-radius: 12px;
            padding: 24px 28px;
            margin-bottom: 16px;
            transition: all 0.3s;
            text-decoration: none;
            display: block;
        }
        
        .result-card:hover {
            border-color: var(--foreground);
            transform: translateY(-2px);
            box-shadow: 0 8px 24px rgba(0, 0, 0, 0.1);
        }
        
        .result-title {
            font-size: 1.25rem;
            font-weight: 600;
            margin-bottom: 8px;
            color: var(--foreground);
            word-wrap: break-word;
        }
        
        .result-description {
            color: var(--muted);
            font-size: 0.9375rem;
            line-height: 1.6;
            margin-bottom: 12px;
            word-wrap: break-word;
        }
        
        .result-slug {
            font-size: 0.875rem;
            color: var(--success);
            font-family: 'Courier New', monospace;
            word-break: break-all;
        }
        
        .no-results {
            text-align: center;
            padding: 60px 24px;
            background: var(--card);
            border: 1px solid var(--border);
            border-radius: 12px;
        }
        
        .no-results-icon {
            width: 64px;
            height: 64px;
            margin: 0 auto 24px;
            opacity: 0.3;
        }
        
        .no-results h3 {
            font-size: 1.5rem;
            margin-bottom: 12px;
        }
        
        .no-results p {
            color: var(--muted);
            font-size: 1rem;
        }
        
        .load-more {
            text-align: center;
            margin-top: 32px;
        }
        
        .load-more-button {
            padding: 14px 32px;
            background: var(--card);
            border: 1px solid var(--border);
            border-radius: 12px;
            color: var(--foreground);
            font-weight: 600;
            font-size: 1rem;
            cursor: pointer;
            transition: all 0.2s;
            text-decoration: none;
            display: inline-block;
        }
        
        .load-more-button:hover {
            background: var(--card-hover);
            border-color: var(--foreground);
            transform: translateY(-2px);
        }
        
        @media (max-width: 768px) {
            .search-page {
                padding: 100px 0 60px;
            }
            
            .search-header {
                margin-bottom: 32px;
            }
            
            .search-header h1 {
                font-size: 1.875rem;
            }
            
            .search-header p {
                font-size: 0.9375rem;
            }
            
            .search-form {
                margin: 0 auto 32px;
            }
            
            .search-input-wrapper {
                flex-direction: column;
                gap: 10px;
            }
            
            .search-input {
                padding: 14px 20px 14px 48px;
                font-size: 0.9375rem;
            }
            
            .search-icon {
                left: 16px;
                width: 18px;
                height: 18px;
            }
            
            .search-button {
                width: 100%;
                padding: 14px 24px;
            }
            
            .search-meta {
                font-size: 0.875rem;
                margin-bottom: 24px;
                padding: 0 4px;
            }
            
            .result-card {
                padding: 18px 20px;
                margin-bottom: 12px;
            }
            
            .result-title {
                font-size: 1.125rem;
            }
            
            .result-description {
                font-size: 0.875rem;
                margin-bottom: 10px;
            }
            
            .result-slug {
                font-size: 0.8125rem;
            }
            
            .no-results {
                padding: 40px 20px;
            }
            
            .no-results-icon {
                width: 48px;
                height: 48px;
                margin-bottom: 16px;
            }
            
            .no-results h3 {
                font-size: 1.25rem;
            }
            
            .no-results p {
                font-size: 0.9375rem;
            }
            
            .load-more-button {
                padding: 12px 28px;
                font-size: 0.9375rem;
            }
            
            .search-icon{
                display: none;
            }
        }
        
        @media (max-width: 480px) {
            .search-page {
                padding: 90px 0 40px;
            }
            
            .search-header h1 {
                font-size: 1.625rem;
            }
            
            .search-header p {
                font-size: 0.875rem;
            }
            
            .search-input {
                padding: 12px 16px 12px 44px;
            }
            
            .result-card {
                padding: 16px 18px;
            }
            
            .result-title {
                font-size: 1.0625rem;
            }
            
            .load-more-button {
                width: 100%;
            }
        }

        .back-link {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            color: var(--muted);
            text-decoration: none;
            font-size: 0.9375rem;
            margin-bottom: 24px;
            transition: color 0.2s;
        }

        .back-link:hover {
            color: var(--foreground);
        }

        .back-link svg {
            transition: transform 0.2s;
        }

        .back-link:hover svg {
            transform: translateX(-4px);
        }
    </style>
</head>
<body>
    <div class="search-page">
        <div class="container">

            <a href="#" onclick="goBack(event)" class="back-link">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <line x1="19" y1="12" x2="5" y2="12"></line>
                    <polyline points="12 19 5 12 12 5"></polyline>
                </svg>
            </a>
            <script>
            function goBack(e) {
                e.preventDefault();
                if (document.referrer !== "" && window.history.length > 1) {
                    window.history.back();
                } else {
                    window.location.href = "/";
                }
            }
            </script>

            <div class="search-header">
                <h1>Search</h1>
                <p>Find articles and pages</p>
            </div>
            
            <form class="search-form" method="get" action="">
                <div class="search-input-wrapper">
                    <svg class="search-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                    </svg>
                    <input 
                        type="text" 
                        name="q" 
                        class="search-input" 
                        placeholder="Search for articles..." 
                        value="<?php echo htmlspecialchars($query); ?>"
                        autocomplete="off"
                        required
                    >
                    <button type="submit" class="search-button">Search</button>
                </div>
            </form>
            
            <?php if ($search_performed): ?>
                <div class="search-meta">
                    <?php if ($total_results > 0): ?>
                        Showing <?php echo $showing_from; ?>-<?php echo $showing_to; ?> of <?php echo $total_results; ?> results for "<strong><?php echo htmlspecialchars($query); ?></strong>"
                    <?php else: ?>
                        No results found for "<strong><?php echo htmlspecialchars($query); ?></strong>"
                    <?php endif; ?>
                </div>
                
                <div class="search-results">
                    <?php if (!empty($results)): ?>
                        <?php foreach ($results as $result): ?>
                            <?php 
                            $safe_slug = preg_replace('/[^a-z0-9\-]/', '', strtolower($result['slug']));
                            if (empty($safe_slug)) continue;
                            ?>
                            <a href="/<?php echo htmlspecialchars($safe_slug, ENT_QUOTES, 'UTF-8'); ?>" class="result-card">
                                <div class="result-title"><?php echo htmlspecialchars($result['title'], ENT_QUOTES, 'UTF-8'); ?></div>
                                <div class="result-description"><?php echo htmlspecialchars($result['description'], ENT_QUOTES, 'UTF-8'); ?></div>
                                <div class="result-slug">/<?php echo htmlspecialchars($safe_slug, ENT_QUOTES, 'UTF-8'); ?></div>
                            </a>
                        <?php endforeach; ?>
                        
                        <?php if ($has_more): ?>
                            <div class="load-more">
                                <a href="?q=<?php echo urlencode($query); ?>&page=<?php echo $page + 1; ?>" class="load-more-button">
                                    Show More Results
                                </a>
                            </div>
                        <?php endif; ?>
                    <?php else: ?>
                        <div class="no-results">
                            <svg class="no-results-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9.172 16.172a4 4 0 015.656 0M9 10h.01M15 10h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            <h3>No results found</h3>
                            <p>Try searching for something else.</p>
                        </div>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script>
        const html = document.documentElement;
        
        function initTheme() {
            let theme = localStorage.getItem('theme');
            if (!theme) {
                const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
                theme = prefersDark ? 'dark' : 'light';
            }
            
            if (theme === 'light') {
                html.setAttribute('data-theme', 'light');
            } else {
                html.removeAttribute('data-theme');
            }
        }
        
        initTheme();
    </script>
</body>
</html>